<?php
namespace SabaiApps\Directories\Component\WordPress\Helper;

use SabaiApps\Directories\Application;
use SabaiApps\Directories\Component\WordPress\Block\IBlock;
use SabaiApps\Directories\Exception;

class BlocksHelper
{
    public function help(Application $application, $useCache = true)
    {
        if (!$useCache
            || (!$blocks = $application->getPlatform()->getCache('wordpress_blocks'))
        ) {
            $blocks = [];
            foreach ($application->InstalledComponentsByInterface('WordPress\IBlocks') as $component_name) {
                if (!$application->isComponentLoaded($component_name)
                    || (!$block_names = $application->getComponent($component_name)->wordpressGetBlockNames())
                ) continue;

                foreach ($block_names as $block_name) {
                    if (!$block = $application->getComponent($component_name)->wordpressGetBlock($block_name)) continue;

                    $blocks[$block_name] = [
                        'component' => $component_name,
                        'info' => $block->wordpressBlockInfo(),
                    ];
                }
            }
            $application->getPlatform()->setCache($blocks, 'wordpress_blocks', 0);
        }

        return $blocks;
    }
    private $_impls = [];

    /**
     * Gets an implementation of SabaiApps\Directories\Component\WordPress\IBlock interface for a given block name
     * @param Application $application
     * @param string $blockName
     * @return IBlock
     */
    public function impl(Application $application, $blockName, $returnFalse = false, $useCache = true)
    {
        if (!isset($this->_impls[$blockName])) {
            if ((!$blocks = $this->help($application, $useCache))
                || !isset($blocks[$blockName]['component'])
                || (!$application->isComponentLoaded($blocks[$blockName]['component']))
            ) {
                if ($returnFalse) return false;
                throw new Exception\UnexpectedValueException(sprintf('Invalid block: %s', $blockName));
            }
            $this->_impls[$blockName] = $application->getComponent($blocks[$blockName]['component'])->wordpressGetBlock($blockName);
        }

        return $this->_impls[$blockName];
    }

    public function init(Application $application, $force = false)
    {
        $blocks = $this->help($application, $force);
        foreach (array_keys($blocks) as $name) {
            $info = $blocks[$name]['info'];
            if (!empty($info['json'])) {
                // Register from block.json
                $package = null;
                if (isset($info['package'])) {
                    $package = $info['package'];
                    unset($info['package']);
                }
                $block_type = $application->getPlatform()->getAssetsDir($package). '/blocks/drts-' . $name;
                unset($info['json']);
            } elseif (!empty($info['editor_script'])) {
                $block_type = 'sabaiapps/drts-' . $name;
                // Register scripts
                if (is_string($info['editor_script'])) {
                    $src = $info['editor_script'];
                    $deps = ['wp-blocks', 'wp-i18n', 'wp-element'];
                } else {
                    $src = $info['editor_script']['src'];
                    $deps = $info['editor_script']['deps'];
                }
                wp_register_script($handle = 'drts-wordpress-block-editor-script-' . $name, $src, $deps);
                $info['editor_script_handles'][] = $handle;
                unset($info['editor_script']);

                foreach (['script', 'view_script'] as $script_type) {
                    if (!isset($info[$script_type])) continue;

                    foreach ((array)$info[$script_type] as $handle => $script) {
                        $handle = 'drts-wordpress-block-' . $script_type . '-' . $handle;
                        wp_register_script($handle, $script['src'], $script['deps']);
                        $info[$script_type . '_handles'][] = $handle;
                    }
                    unset($info[$script_type]);
                }

                // Register styles
                foreach (['editor_style', 'style'] as $style_type) {
                    if (empty($info[$style_type])) continue;

                    foreach ((array)$info[$style_type] as $handle => $style) {
                        $handle = 'drts-wordpress-block-' . $style_type . '-' . $handle;
                        wp_register_style($handle, $style['src'], $style['deps']);
                        $info[$style_type . '_handles'][] = $handle;
                    }
                    unset($info[$style_type]);
                }
            } else {
                $application->logError('Property json or entity_script missing for block: ' . $name);
                unset($blocks[$name]);
                continue;
            }

            // Set default properties
            $info += [
                'version' => Application::VERSION,
                'api_version' => 3,
                'category' => 'drts',
                'icon' => 'smiley',
                'textdomain' => 'directories',
            ];

            if (!isset($info['render'])
                || $info['render'] !== false
            ) {
                $info['render_callback'] = [$application, 'WordPress_Blocks_renderBlock'];
            }

            register_block_type($block_type, $info);
        }
        if (!empty($blocks)) {
            add_action('enqueue_block_editor_assets', [$application, 'WordPress_Blocks_enqueueBlockEditorAssets']);
        }

        // Add custom block category
        add_filter(version_compare($GLOBALS['wp_version'], '5.8', '>=') ? 'block_categories_all' : 'block_categories', function($categories) {
            $categories[] = [
                'slug'  => 'drts',
                'title' => 'Directories',
            ];
            return $categories;
        });
    }

    public function enqueueBlockEditorAssets(Application $application)
    {
        // Enqueue frontend CSS files.
        $application->getPlatform()->loadMainCoreCss();

        $blocks = $this->help($application);
        foreach (array_keys($blocks) as $name) {
            if (!$block = $this->impl($application, $name, true)) continue;
            if (!empty($blocks[$name]['info']['json'])) {
                $handle = 'sabaiapps-drts-' . $name . '-editor-script';
            } else {
                $handle = 'drts-wordpress-block-editor-script-' . $name;
            }
            if ($_vars = $block->wordpressBlockEditorVars()) {
                wp_localize_script($handle, 'drts_wordpress_block_editor_vars_' . $name, $_vars);
            }
            // Allow block to do something
            $block->wordpressBlockOnEnqueueEditorAssets($handle);
        }
    }

    public function renderBlock(Application $application, $attributes, $content)
    {
        // Render callback called from the editor or the frontend?
        $is_editor = defined( 'REST_REQUEST' ) && REST_REQUEST;

        if (empty($attributes['block'])) {
            $error = 'Attribute "block" may not be empty: ' . print_r($attributes, true);
        } elseif ((strpos($attributes['block'], 'sabaiapps/drts-') !== 0)
            || (!$block_name = substr($attributes['block'], strlen('sabaiapps/drts-')))
        ) {
            $error = 'Invalid block name: ' . $attributes['block'];
        } else {
            try {
                return $this->impl($application, $block_name)->wordpressBlockRender((array)$attributes, $content, $is_editor);
            } catch (Exception\IException $e) {
                $error = $e->getMessage();
            }
        }

        if ($is_editor) return $error;

        $application->logError($error);
        return '';
    }
}